function [Cij] = dipoles_coupling_matrix(k, nji, Rij, alphae, alphah,...
    alphaeh, eps0, mu0)

% Calculation of a 6x6 coupling matrix C between two magneto-electric 
% dipoles. This coupling matrix is a submatrix of the 6*Ntx6*Nt global Cg 
% matrix calculated in the  function global_interaction_matrix, where Nt is 
% the number of dipoles considered in the calculation. This function is 
% only designed for being called by the function global_interaction_matrix.

%%%%%% INPUT ARGUMENTS %%%%%%
%%% k:   i   ncident wave number                                       %%%
%%% nji:     1x3 unitary vector of the distance from the j-th dipole   %%%
%%%          to the i-th dipole                                        %%%
%%% Rij:     the module of the latter distance                         %%%
%%% alphae:  electric polarizability of the j-th dipole                %%%
%%% alphah:  magnetic polarizability of the j-th dipole                %%%
%%% alphaeh: mixed magnetic-electric polarizability of the j-th dipole %%%
%%% eps0:    surrounding permittivity                                  %%%
%%% mu0:     surrounding permeability                                  %%%

%%%%%% OUTPUT ARGUMENTS %%%%%%
%%% Cij: 6x6 coupling matrix between the two dipoles %%%

% For more information see: Rosales, S. A., Albella, P., González, F., 
% Gutierrez, Y., & Moreno, F. (2021). CDDA: extension and analysis of the 
% discrete dipole approximation for chiral systems. Optics Express, 29(19), 
% 30020-30034. Especially the suplementary information.


% Auxiliary constants %
Z0  = sqrt(mu0/eps0);
Aij = (k^2- 1/Rij^2 + 1i*k/Rij)*(exp(1i*k*Rij)/(4*pi*Rij*eps0));
Bij = (3/Rij^2 - k^2 - 1i*3*k/Rij)*(exp(1i*k*Rij)/(4*pi*Rij*eps0));
Dij = (k^2+1i*k/Rij)*(Z0*exp(1i*k*Rij)/(4*pi*Rij));

% Auxiliary matrices %
C = Aij*eye(3) + Bij*(nji')*nji;
F = Dij*[   0     -nji(3)    nji(2)
          nji(3)     0      -nji(1)
         -nji(2)   nji(1)      0   ];

% Polarizability input is for one wavelength, one dipole %
alphae  = squeeze(alphae);
alphah  =  squeeze(alphah);
alphaeh = squeeze(alphaeh);

% Coupling matrix definition %
Cij(1:3,1:3) =  C*alphae - 1i*F*alphaeh;
Cij(1:3,4:6) = -F*alphah - 1i*(mu0)*C*alphaeh;
Cij(4:6,1:3) =  (1/mu0)*F*alphae + 1i*(eps0)*C*alphaeh;
Cij(4:6,4:6) =   (eps0)*C*alphah - 1i*F*alphaeh;
     
end


